local bids = {}
local proxyBids = {}
local classes = {}
local realclasses = {}
local bidItem = nil
local maxBid = 0
local bidsChanged = true
local bidders = {}

local showOptions = {
	["name"] = "Name",
	["class"] = "Class",
	["num"] = "Number of bidders",
	["none"] = "Anonymous",
	["silent"] = "Silent"
}

local new, newHash, newSet, del
do
	local list = setmetatable({}, {__mode='k'})
	
	function new(...)
		local t = next(list)
		if t then
			list[t] = nil
			for i = 1, select('#', ...) do
				t[i] = select(i, ...)
			end
			return t
		else
			return {...}
		end
	end
	function del(t)
		setmetatable(t, nil)
		for k in pairs(t) do
			t[k] = nil
		end
		t[''] = true
		t[''] = nil
		list[t] = true
		return nil
	end
end

local options = {
	type = "group",
	args = {
		bid = {
			type = "text",
			name = "Open bidding on...",
			desc = "Open bidding on...",
			get = function() return bidItem or "" end,
			set = function(v)
				WhisperBid:OpenBidding(v)
			end,
			validate = function(v)
				return v:match("|Hitem:") ~= nil
			end,
			usage = "<item link>"
		},
		close = {
			type = "execute",
			name = "Close bidding",
			desc = "Close bidding",
			disabled = function() return bidItem == nil end,
			func = function() WhisperBid:CloseBidding() end
		},
		announce = {
			type = "execute",
			name = "Announce bids",
			desc = "Announce bids",
			disabled = function() return bidItem == nil end,
			func = function() WhisperBid:Announce() end
		},
		show = {
			type = "text",
			name = "Show...",
			desc = "Show...",
			get = function() return WhisperBid.db.profile.ShowInfo end,
			set = function(v)
				WhisperBid.db.profile.ShowInfo = v
			end,
			validate = showOptions,
			usage = "<option>"
		},
		remove = {
			type = "text",
			name = "Remove Bid",
			desc = "Remove Bid",
			get = function() return "" end,
			set = function(v) WhisperBid:RemoveBidder(v) end,
			validate = bidders
		},
		timeout = {
			type = "range",
			name = "Bid Timeout",
			desc = "Close bidding automatically after a number of seconds (0 to disable)",
			min = 0,
			max = 300,
			step = 5,
			isPercent = false,
			get = function() return WhisperBid.db.profile.Timeout end,
			set = function(v) WhisperBid.db.profile.Timeout = v end
		},
		pause = {
			type = "toggle",
			name = "Pause",
			desc = "Pause",
			disabled = function() return not bidItem end,
			get = function() return WhisperBid.paused end,
			set = function(v) WhisperBid:Pause(v) end
		},
		warnings = {
			type = "toggle",
			name = "Warn Time Remaining",
			desc = "Periodically warn the raid about the time remaining for bidding",
			get = function() return WhisperBid.db.profile.TimeWarnings end,
			set = function(v) WhisperBid.db.profile.TimeWarnings = v end
		},
		suppress = {
			type = "toggle",
			name = "Suppress Whispers",
			desc = "Suppress automated outbound whispers",
			get = function() return WhisperBid.db.profile.SuppressOutbound end,
			set = function(v) WhisperBid.db.profile.SuppressOutbound = v end
		},
		raidlist = {
			type = "execute",
			name = "Get Raid List",
			desc = "Get Raid List",
			func = function() WhisperBid.CopyRaidList() end,
			disabled = function() return GetNumRaidMembers() == 0 end
		}
	}
}

local Tablet = AceLibrary:HasInstance("Tablet-2.0") and AceLibrary("Tablet-2.0")
local dontUseTablet = false

WhisperBid = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "FuBarPlugin-2.0", "AceDB-2.0")
WhisperBid:RegisterChatCommand({"/wb"}, options, "WHISPERBID")
WhisperBid.OnMenuRequest = options
WhisperBid:RegisterDB("WhisperBidDB", "WhisperBidDBPC", "profile")
WhisperBid:RegisterDefaults('profile', {
	ShowInfo = "class",
	Timeout = 30,
	TimeWarnings = true,
	SuppressOutbound = true
})

local lastBidTime = 0

local function getDistribution()
	if GetNumRaidMembers() > 0 then return "RAID" end
	if GetNumPartyMembers() > 0 then return "PARTY" end
end

-- From AceConsole
local function confirmPopup(message, func, ...)
	if not StaticPopupDialogs["ACECONSOLE20_CONFIRM_DIALOG"] then
		StaticPopupDialogs["ACECONSOLE20_CONFIRM_DIALOG"] = {}
	end
	local t = StaticPopupDialogs["ACECONSOLE20_CONFIRM_DIALOG"]
	for k in pairs(t) do
		t[k] = nil
	end
	t.text = message
	t.button1 = ACCEPT or "Accept"
	t.button2 = CANCEL or "Cancel"
	t.OnAccept = function()
		func(unpack(t))
	end
	for i = 1, select('#', ...) do
		t[i] = select(i, ...)
	end
	t.timeout = 0
	t.whileDead = 1
	t.hideOnEscape = 1
	
	StaticPopup_Show("ACECONSOLE20_CONFIRM_DIALOG")
end

function WhisperBid:OnInitialize()
	self:RegisterEvent("CHAT_MSG_WHISPER")
	self.hasIcon = [[Interface\Icons\INV_Enchant_ShardNexusLarge]]
	self.hasNoColor  = true
	self.clickableTooltip  = true
	if not Tablet or dontUseTablet then
		self.blizzardTooltip = true
	end
	self.hideWithoutStandby = true
	self.p = self.db.profile
	self.WhisperPrefix = "[WB]"
end

function WhisperBid.FilterChatMessage(event)
	if bidItem and event == "CHAT_MSG_WHISPER" and WhisperBid.db.profile.SuppressOutbound and arg1 and arg1 ~= "" and arg1:sub(1, string.len(WhisperBid.WhisperPrefix)) == WhisperBid.WhisperPrefix then
		return
	end
	return WhisperBid.Original_ChatFrame_MessageEventHandler(event)
end

local function SendSuppressedWhisper(what, who)
	SendChatMessage(WhisperBid.WhisperPrefix .. " " .. what, "WHISPER", nil, who)
end

function WhisperBid:OnEnable()
	self:SetIcon([[Interface\Icons\INV_Enchant_ShardNexusLarge]])
	self.Original_ChatFrame_MessageEventHandler = _G.ChatFrame_MessageEventHandler
	_G.ChatFrame_MessageEventHandler = self.FilterChatMessage
end

function WhisperBid:OnDisable()
	_G.ChatFrame_MessageEventHandler = self.Original_ChatFrame_MessageEventHandler
end

function WhisperBid:UpdateTimeout()
	if self.paused then return end
	if timeout == 0 then return end
	local timeout = self.db.profile.Timeout
	self:Update()
	local remaining  = timeout - (GetTime() - lastBidTime)
	if timeout >= 15 and self.db.profile.TimeWarnings then
		local halfTime = math.ceil(self.db.profile.Timeout / 10) * 5
		local quarterTime = math.ceil(self.db.profile.Timeout / 20) * 5
		if math.floor(remaining) == halfTime or math.floor(remaining) == quarterTime then
			local dist = getDistribution():lower()
			SendChatMessage(("Closing bidding on %s in %s"):format(bidItem, SecondsToTime(remaining)), dist)
		end
	end
	if remaining <= 0 then
		local dist = getDistribution():lower()
		SendChatMessage(("No bids for %s, automatically closing bidding on %s"):format(SecondsToTime(self.db.profile.Timeout), bidItem), dist)
		self:CloseBidding()
	end
end

function WhisperBid:OpenBidding(item)
	if bidItem then
		self:Print("Can't open bidding on %s: there is a running bid on %s. Use /wb close to close bidding on the previous item.", item, bidItem)
		return
	end
	
	local lootmethod, masterlooterID = GetLootMethod()
	if GetNumRaidMembers() > 0 and (not IsRaidOfficer() and not IsRaidLeader() and not (lootmethod == "master" and masterlooterID == 0)) then
		self:Print("You must be a raid officer or the Master Looter to open bidding on %s.", item)
		return
	end
		
	local dist = "PARTY"
	bidItem = item
	maxBid = 0
	lastBidTime = GetTime()
	self:ScheduleRepeatingEvent("WhisperBidTimeout", self.UpdateTimeout, 1, self)
	self:ClearBids()
	if GetNumRaidMembers() > 0 then
		dist = "RAID"
	end
	if self.db.profile.Timeout > 0 then
		SendChatMessage(("Bids open on " .. item .. ". Whisper \"bid <number>\" to me to register your bid, or \"proxy <number>\" to register a proxy bid."), dist)
	else
		SendChatMessage(("Bids open on " .. item .. ". Whisper \"bid <number>\" to me to register your bid, or \"proxy <number>\" to register a proxy bid. Bidding will close in %s."):format(SecondsToTime(self.db.profile.Timeout)), dist)
	end
	self:Update()
end

local function func(a,b)
	if not a then return false end
	if not b then return true end
	return a[2] > b[2]
end

local function clickfunc(who)
	WhisperBid:RemoveBidder(who)
end

local remainingTime = 0
WhisperBid.paused = false
function WhisperBid:Pause(value)
	if WhisperBid.paused or not value then
		lastBidTime = GetTime() - (WhisperBid.db.profile.Timeout - remainingTime)
	end
	remainingTime = lastBidTime + WhisperBid.db.profile.Timeout - GetTime()
	if value ~= nil then
		WhisperBid.paused = value
	else
		WhisperBid.paused = not WhisperBid.paused
	end
end

local bidInfo = {}
local default = {r = 1, g = 1, b = 1}
function WhisperBid:OnTooltipUpdate()
	if bidsChanged then
		for i = 1, #bidInfo do
			del(tremove(bidInfo))
		end
		for k, v in pairs(bids) do
			tinsert(bidInfo, new(k, v))
		end
		bidsChanged = false
		table.sort(bidInfo, func)
	end
	if not Tablet or dontUseTablet then
		self:BlizzardTooltip()
	else
		local cat = Tablet:AddCategory(
			'columns', 2,
			'child_textR', 1,
			'child_textG', 1,
			'child_textB', 0,
			'child_text2R', 1,
			'child_text2G', 1,
			'child_text2B', 1
		)
		if bidItem then
			cat:AddLine( 'text', "Bidding on:", 'text2', bidItem)
			if self.db.profile.Timeout > 0 then 
				if WhisperBid.paused then
					cat:AddLine( 'text', "[Paused]", "func", WhisperBid.Pause, "arg1", self, "arg2", false)
				else
					cat:AddLine( 'text', ("%s remaining"):format(SecondsToTime(WhisperBid.db.profile.Timeout - (GetTime() - lastBidTime))), "func", WhisperBid.Pause, "arg1", self, "arg2", true)
				end
			end
			for k, line in ipairs(bidInfo) do
				local t = RAID_CLASS_COLORS[realclasses[line[1]]] or default
				cat:AddLine( 'text', ("|cff%02x%02x%02x%s|r"):format(t.r * 255, t.g * 255, t.b * 255, line[1]), 'func', clickfunc, 'arg1', line[1], 'text2', line[2])
			end
		else
			cat:AddLine( 'text', "No item open for bids", 'text2', "")
		end
	end
end

function WhisperBid:BlizzardTooltip()
	if bidItem then
		GameTooltip:AddLine(("Bidding on: %s"):format(bidItem))
		if self.db.profile.Timeout > 0 then 
			if WhisperBid.paused then
				GameTooltip:AddLine("[Paused]")
			else
				GameTooltip:AddLine(("%s remaining"):format(SecondsToTime(WhisperBid.db.profile.Timeout - (GetTime() - lastBidTime))))
			end
		end
		for k, line in ipairs(bidInfo) do
			local t = RAID_CLASS_COLORS[realclasses[line[1]]]
			GameTooltip:AddLine(("%s - |cff%02x%02x%02x%s|r"):format(line[1], t.r * 255, t.g * 255, t.b * 255, line[2]))
			if i == 10 then break end
		end
	else
		GameTooltip:AddLine("Hint: type /wb bid to begin bidding on an item")
	end
end

function WhisperBid:CHAT_MSG_WHISPER(msg, author, language, status)
	if not bidItem then return end
	msg = msg:lower()
	local amount = tonumber(msg:match("bid (%d+)"))
	local proxy_amount = tonumber(msg:match("proxy (%d+)"))
	if not amount and not proxy_amount then 
		amount = tonumber(msg:match("^(%d+)"))
	end
	if amount then
		self:HandleBid(author, amount)
	elseif proxy_amount then
		self:HandleProxyBid(author, proxy_amount)
	elseif msg:match("^bid") then
		SendSuppressedWhisper("Please send bids in the form \"bid #\". Example: bid 4", author)
	end
end

local function removeBidder(self, who)
	local dist = getDistribution()	
	bids[who] = nil
	for i = 1, #bidders do
		if who == bidders[i] then
			tremove(bidders, i)
			break
		end
	end
	maxBid = 0
	for k,v in pairs(bids) do
		if v > maxBid then
			maxBid = v
		end
	end
	bidsChanged = true
	SendChatMessage(("Bid on %s removed [%s]"):format(bidItem, classes[who]), dist)
	self:Update()
	self:Announce()
end

function WhisperBid:RemoveBidder(who)
	confirmPopup(("Remove %s's bid?"):format(who), removeBidder, self, who)
end

function WhisperBid:HandleProxyBid(from, bid)
	if not bidItem then return end
	local BID_MAX = 1000000
	if bid > BID_MAX then
		SendSuppressedWhisper(("Your bid of %s is greater than the maximum allowable bid of %s"):format(bid, BID_MAX), from)
		return
	end

	if bid < maxBid then
		SendSuppressedWhisper(("You may not set your proxy bid lower than the current max bid of %s"):format(maxMid), from)
		return
	end
	
	proxyBids[from] = bid
	
	self:UpdateProxyBids()
end

local bidAmts = {}
function WhisperBid:UpdateProxyBids()
	for i = 1, #bidAmts do
		tremove(bidAmts)
	end
	local maxBidder = nil
	local result
	for k, v in pairs(proxyBids) do
		if v >= maxBid then
			tinsert(bidAmts, v)
		end
	end
	table.sort(bidAmts)
	local highCommonBid = 0
	if #bidAmts >= 2 then
		highCommonBid = bidAmts[#bidAmts-1]
	elseif #bidAmts == 1 then
		highCommonBid = maxBid
	else
		return
	end
	for k, v in pairs(proxyBids) do
		if v >= highCommonBid and (not bids[k] or bids[k] <= highCommonBid) then
			if v > highCommonBid then
				self:HandleBid(k, highCommonBid + 1, true)
			elseif not bids[k] or bids[k] < highCommonBid then
				self:HandleBid(k, highCommonBid, true)
			else
				-- self:Print("%s == %s, but standing bid for %s is %s. Not bidding.", v, highCommonBid, k, bids[k])
			end
		end
	end
end

local bidCheckers = {}
function WhisperBid:RegisterBidCheck(addon, func)
	bidCheckers[addon] = func
end

function WhisperBid:HandleBid(from, bid, silent)
	local BID_MAX = 1000000
	if bid > BID_MAX then
		if not silent then
			SendSuppressedWhisper(("Your bid of %s is greater than the maximum allowable bid of %s"):format(bid, BID_MAX), from)
		end
		return
	end
	local raidMembers = GetNumRaidMembers()
	local partyMembers = GetNumPartyMembers()
	local members = (raidMembers > 0 and raidMembers) or partyMembers
	local dist = getDistribution():lower()
	for i = 1, members do
		local unit = nil
		if UnitName(dist .. i) == from then
			unit = dist .. i
		elseif UnitName("player") == from then
			unit = "player"
		end
		if unit then
			for k,v in pairs(bidCheckers) do
				local result, msg
				if type(v) == "table" then
					result, msg = v(k, from, bid)
				else
					result, msg = v(from, bid)
				end
				if not result then
					SendSuppressedWhisper(("Unable to register your bid of %s: %s"):format(bid, msg or "External bid check failed."), from)
					return false
				end
			end
			classes[from] = UnitClass(unit)
			realclasses[from] = select(2, UnitClass(unit))
			if not bids[from] then
				tinsert(bidders, from)
			end
			if bids[from] and bid <= bids[from] then
				if not silent then
					SendSuppressedWhisper(("Your bid of %s is not greater than your previous bid of %s"):format(bid, bids[from]), from)
				end
				return false
			elseif bid < maxBid then
				if not silent then
					SendSuppressedWhisper(("Your bid of %s is less than the current max bid of %s"):format(bid, maxBid), from)
				end
				return false
			elseif bid == maxBid then
				if not silent then
					SendSuppressedWhisper(("Your bid of %s ties than the current max bid of %s"):format(bid, maxBid), from)
				end
				bids[from] = bid
				lastBidTime = GetTime()
				bidsChanged = true
				self:TriggerEvent("WhisperBid_NewTieBid", bidItem, from, bid)
				self:Update()
				self:Announce()
			else
				if self.db.profile.ShowInfo == "silent" and not silent then
					SendSuppressedWhisper(("Your bid of %s on %s has been accepted"):format(bid, bidItem), from)
				end
				maxBid = bid
				bidsChanged = true
				lastBidTime = GetTime()
				bids[from] = bid
				self:TriggerEvent("WhisperBid_NewHighBid", bidItem, from, bid)
				self:Update()
				self:Announce()
			end
			if not silent then
				self:UpdateProxyBids()
			end
			return true
		end
	end	
end

function WhisperBid:ClearBids()
	for k, v in pairs(bids) do
		bids[k] = nil
	end
	for k, v in pairs(proxyBids) do
		proxyBids[k] = nil
	end
	for k, v in pairs(bidders) do
		bidders[k] = nil
	end
	bidsChanged = true
end

local bidClasses = {}
function WhisperBid:Announce()
	local showInfo = self.p.ShowInfo 
	if showInfo == "silent" then return end 
	
	local dist = getDistribution()

	for i = 1, #bidClasses do
		tremove(bidClasses)
	end
	for k, v in pairs(bids) do
		if v == maxBid then
			if showInfo == "class" then
				tinsert(bidClasses, classes[k])
			else
				tinsert(bidClasses, k)
			end
		end
	end
	table.sort(bidClasses)
	if #bidClasses > 1 then
		local info = table.concat(bidClasses, ", ")
		if showInfo == "num" then
			info = #bidClasses .. " bidders"
		elseif showInfo == "none" then
			info = "anonymous"
		end
		SendChatMessage(("High bid on %s is tied at %s [%s]"):format(bidItem, maxBid, info), dist)
	elseif #bidClasses == 1 then
		local info = table.concat(bidClasses, ", ")
		if showInfo == "num" then
			info = "1 bidder"
		elseif showInfo == "none" then
			info = "anonymous"
		end
		SendChatMessage(("High bid on %s is %s [%s]"):format(bidItem, maxBid, info), dist)
	else
		SendChatMessage(("No bids on %s"):format(bidItem), dist)
	end
end

local bidNames = {}
function WhisperBid:CloseBidding()
	local dist = getDistribution()

	self:CancelScheduledEvent("WhisperBidTimeout")
	
	for i = 1, #bidNames do
		tremove(bidNames)
	end
	
	for k, v in pairs(bids) do
		if v == maxBid then
			tinsert(bidNames, k)
		end
	end
	table.sort(bidNames)
	if #bidNames > 1 then
		SendChatMessage(("%d-way tie on %s for %s points: %s, please roll."):format(#bidNames, bidItem, maxBid, table.concat(bidNames, ", ")), dist)
		self:TriggerEvent("WhisperBid_ClosedTie", bidItem, bid, unpack(bidNames))
	elseif #bidNames == 1 then
		SendChatMessage(("%s wins %s for %s points!"):format(bidNames[1], bidItem, maxBid), dist)
		self:TriggerEvent("WhisperBid_Closed", bidItem, bid, unpack(bidNames))
	else
		SendChatMessage(("No bids on %s"):format(bidItem), dist)
		self:TriggerEvent("WhisperBid_ClosedNoBids", bidItem)
	end	
	bidItem = nil
	bidsChanged = true
	self:Update()
end

local members = {}
function WhisperBid.CopyRaidList()
	if not StaticPopupDialogs["WHISPER_BID_RAID_LIST_COPY"] then
	    -- Copied from the UrlCopy Prat module
	StaticPopupDialogs["WHISPER_BID_RAID_LIST_COPY"] = {
		text = "Members : %s",
		button2 = TEXT(ACCEPT),
		hasEditBox = 1,
		hasWideEditBox = 1,
		showAlert = 1, -- HACK : it"s the only way I found to make de StaticPopup have sufficient width to show WideEditBox :(

		OnShow = function()
			local editBox = getglobal(this:GetName().."WideEditBox")
			editBox:SetText(format(WhisperBid.members))
			editBox:SetFocus()
			editBox:HighlightText(0)
			editBox:SetMaxLetters(500)

			local button = getglobal(this:GetName().."Button2")
			button:ClearAllPoints()
			button:SetWidth(200)
			button:SetPoint("CENTER", editBox, "CENTER", 0, -30)

			getglobal(this:GetName().."AlertIcon"):Hide()  -- HACK : we hide the false AlertIcon
		end,

		OnHide = function() end,
		OnAccept = function() end,
		OnCancel = function() end,
		EditBoxOnEscapePressed = function() this:GetParent():Hide() end,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1
	};
	end
	for i = 1, #members do
		tremove(members)
	end
	for i = 1, GetNumRaidMembers() do
		local raidMember = UnitName("raid" .. i)
		tinsert(members, raidMember)
	end
	table.sort(members)
	WhisperBid.members = table.concat(members, ", ")
	StaticPopup_Show ("WHISPER_BID_RAID_LIST_COPY", table.concat(members, ", "));
end

--[[
SlashCmdList["RaidListCopy_Slash_Command"] = WhisperBid.CopyRaidList
SLASH_RaidListCopy_Slash_Command = "/raidlist"
]]--
